<?php

namespace Mdr\Services;

class MdrPay
{

    public function get_transaction_data()
    {
        return $this->transaction_data;
    }

    public function verify(int $transaction_id, $sign)
    {
        global $wpdb, $mdr_pay_message_error;

        if ($sign != $this->get_transaction_sign($transaction_id)) {
            $mdr_pay_message_error = 'لینک بازگشت از پرداخت نامعتبر می باشد.';
            return;
        }

        $transaction_tb = new DBAccess($wpdb->prefix . "mdr_transactions");
        $transaction = $transaction_tb->get_row(['id' => $transaction_id]);

        if (!$transaction) {
            $mdr_pay_message_error = 'تراکنش یافت نشد.';
            return;
        }

        if ($transaction['status'] != 'pending') {
            $mdr_pay_message_error = 'این تراکنش قبلا پردازش شده است.';
            return;
        }

        $this->transaction = $transaction;

        $gateway = $this->get_gateway($transaction['gateway']);

        if (!$gateway) {
            $mdr_pay_message_error = 'درگاه مورد نظر یافت نشد.';
            return;
        }

        $request = new $gateway['class']($gateway);
        $request = $request->verify($transaction['authority']);

        if (!$request['success']) {
            $update_transaction = $transaction_tb->update(['id' => $transaction_id], [
                'status' => 'failed',
            ]);
            $mdr_pay_message_error = 'پرداخت ناموفق بود. کد پیگیری: ' . $this->transaction['authority'];
            return;
        }

        $update_data = [
            'status' => 'paid',
            'paid_at' => current_time('mysql')
        ];

        if (empty($this->transaction['user_id'])) {
            $inputs = json_decode($this->transaction['data'], true);
            $user = mdr_get_or_create_user($inputs['phone'], $inputs['full_name']);
            $update_data['user_id'] = $user->ID;
            $this->transaction['user_id'] = $user->ID;
        }

        $update_transaction = $transaction_tb->update(['id' => $transaction_id], $update_data);

        if (!$update_transaction) {
            $mdr_pay_message_error = 'خطایی در روند پردازش تراکنش رخ داده لطفا با پشتیبانی تماس بگیرید. شماره پیگیری: ' . $transaction['authority'];
            return;
        }

        $this->transaction['status'] = 'paid';
        $this->transaction['paid_at'] = current_time('mysql');

        //create order process
        $products = json_decode($transaction['products'], true);

        if (!empty($products)) {
            $order = $this->create_order();

            if (!$order) {
                $mdr_pay_message_error = 'خطایی در روند ایجاد سفارش رخ داده لطفا با پشتیبانی تماس بگیرید. شماره پیگیری: ' . $transaction['authority'];
                return;
            }

            $update_transaction = $transaction_tb->update(['id' => $transaction_id], [
                'order_id' => $order->get_id(),
            ]);

        }

        //show success page
        //TODO
    }

    private function create_order()
    {

        if (empty($this->transaction)) {
            return false;
        }

        $inputs = json_decode($this->transaction['data'], true);
        $user = mdr_get_user($inputs['phone']);

        $new_order = [
            'customer_id' => $this->transaction['user_id'],
            'created_via' => 'mdr_theme_pay',
        ];

        $order = wc_create_order($new_order);
        $products = json_decode($this->transaction['products'], true);

        foreach ($products as $product_id => $price) {

            $product = wc_get_product($product_id);

            if (!$product) {
                continue;
            }

            if ($price == -1) {
                $price = $product->get_price();
            }

            $price_amount = [
                'total' => $price,
            ];
            $order->add_product($product, 1, $price_amount);

        }

        $utm = json_decode($this->transaction['utm'], true);

        if (is_array($utm) && !empty($utm)) {

            foreach ($utm as $meta_key => $meta_value) {
                $order->add_meta_data(sanitize_text_field($meta_key), sanitize_text_field($meta_value), true);
            }

        }

        $order->add_meta_data('_billing_phone', $user->user_login, true);
        $order->add_meta_data('_billing_email', $user->user_email, true);
//        $order->add_order_note('خرید تکی محصول');

        $address = [
            'first_name' => $user->display_name,
        ];
        $order->set_address($address, 'billing');
        $order->calculate_totals();
        $order->update_status('wc-completed', 'سفارش اضافه شده توسط خرید تکی محصول کاربر');

        return $order;
    }

    public function set_data(array $inputs, array $data)
    {

        foreach ($inputs as $key => $value) {
            $this->inputs[$key] = $value;
        }

        if (!$this->check_inputs()) {
            return ['errors' => $this->errors];
        }

        foreach ($data as $key => $value) {
            $this->transaction_data[$key] = $value;
        }

        if (!$this->check_data()) {
            return ['errors' => $this->errors];
        }

        if (!$this->complete_create_records_after_check_data()) {
            return ['errors' => $this->errors];
        }

        return true;

    }

    public function getPayLink(): array
    {
        global $wpdb;

        if (!empty($this->errors)) {
            return [
                'success' => false,
                'message' => 'خطایی در اطلاعات رخ داده',
                'data' => ['errors' => $this->errors],
            ];
        }

        $transaction_tb = new DBAccess($wpdb->prefix . 'mdr_transactions');
        $transaction_id = $transaction_tb->insert([
            'user_id' => $this->transaction_data['user_id'] ?? '',
            'data' => json_encode($this->inputs),
            'products' => json_encode($this->transaction_data['products'] ?? json_encode([])),
            'ip' => $this->transaction_data['ip'] ?? '',
            'utm' => json_encode($this->transaction_data['utm'] ?? json_encode([])),
            'landing_url' => $this->transaction_data['landing_url'],
            'status' => 'pending',
            'amount' => $this->transaction_data['amount'] ?? 0,
            'currency' => $this->transaction_data['currency'] ?? $this->currency,
            'gateway' => $this->transaction_data['gateway'],
            'created_at' => current_time('mysql'),
        ]);

        if (!$transaction_id) {
            return [
                'success' => false,
                'message' => 'خطایی در ایجاد رکورد پرداختی رخ داده.'
            ];
        }

        $callback = $this->get_callback($transaction_id);

        if ($this->transaction_data['amount'] > 0) {
            //request
            if (!class_exists($this->gateway['class'])) {
                return [
                    'success' => false,
                    'message' => 'درگاه انتخاب شده به طور کامل راه اندازی نشده است.'
                ];
            }

            $request = new $this->gateway['class']($this->gateway);
            $request = $request->request([
                'callback' => $callback,
                'amount' => $this->transaction_data['amount'],
                'currency' => $this->transaction_data['currency'] ?? $this->currency,
                'order_id' => $transaction_id,
                'phone' => $this->inputs['phone'],
                'description' => $this->transaction_data['description'] ?? '',
            ]);

            if ($request['success']) {
                $update_transaction = $transaction_tb->update(['id' => $transaction_id], [
                    'callback' => $callback,
                    'authority' => $request['data']['authority'],
                ]);
            }

            return $request;

        } else {

            $update_transaction = $transaction_tb->update(['id' => $transaction_id], [
                'callback' => $callback,
            ]);

            return [
                'success' => true,
                'message' => 'بدون پرداختی',
                'data' => ['link' => $callback],
            ];

        }

    }

    public function get_callback($transaction_id)
    {

        if (isset($this->transaction_data['callback']) && !empty($this->transaction_data['callback'])) {

            return $this->transaction_data['callback'];
        } else {
            $query = [
                'tra' => $transaction_id,
                'sign' => $this->get_transaction_sign($transaction_id),
            ];

            return $this->transaction_data['landing_url'] . '?' . http_build_query($query);
        }

    }

    public function get_transaction_sign($transaction_id)
    {
        return substr(md5(AUTH_KEY . str_rot13(sha1($transaction_id))), 5, 5);
    }

    public function check_inputs()
    {

        $error = true;

        $this->inputs['phone'] = mdr_sanitize_mobile($this->inputs['phone'] ?? '');

        if (!mdr_is_phone($this->inputs['phone'])) {
            $this->errors['input'][] = [
                'name' => 'phone',
                'message' => 'شماره وارد شده صحیح نمی باشد.'
            ];
            $error = false;
        }

        if (empty($this->inputs['full_name'] ?? '') || strlen($this->inputs['full_name'] ?? '') < 3) {
            $this->errors['input'][] = [
                'name' => 'full_name',
                'message' => "نام وارد شده کامل نیست."
            ];
            $error = false;
        }

        return $error;
    }

    public function check_data()
    {

        //check important parameters
        if (!isset($this->transaction_data['products'])) {
            $this->errors['data'][] = [
                'name' => 'product',
                'message' => 'پارامتر محصولات یافت نشد.'
            ];
        }

        if (!isset($this->transaction_data['landing_url'])) {
            $this->errors['data'][] = [
                'name' => 'landing_url',
                'message' => 'نام لندینگ یافت نشد.'
            ];
        }

        //check all parameters if exists
        foreach ($this->transaction_data as $key => $value) {

            if (!in_array($key, ['products', 'user_id', 'utm', 'ip', 'landing_url', 'callback', 'currency', 'gateway', 'description'])) {
                unset($this->transaction_data[$key]);
                continue;
            }

            if ($key == 'products' && is_array($value)) {

                foreach ($value as $product_id => $price) {
                    $product = wc_get_product($product_id);

                    if (!$product) {
                        $this->errors['data'][] = [
                            'name' => 'product',
                            'message' => 'محصول مورد نظر یافت نشد.'
                        ];

                        return false;
                    }

                    $amount = $this->transaction_data['amount'] ?? 0;

                    if (!$price || $price < 0) {
                        $amount += intval($product->get_price());
                    } else {
                        $amount += $price;
                    }

                    $this->transaction_data['amount'] = $amount;

                }

            } elseif ($key == 'utm' && is_array($value)) {

                if (isset($this->transaction_data['utm']['utm_source'])) {
                    $this->transaction_data['utm']['utm_source'] = sanitize_text_field($this->transaction_data['utm']['utm_source']);
                }

                if (isset($this->transaction_data['utm']['utm_campaign'])) {
                    $this->transaction_data['utm']['utm_campaign'] = sanitize_text_field($this->transaction_data['utm']['utm_campaign']);
                }

                $this->transaction_data['utm'] = json_encode($this->transaction_data['utm']);
            } elseif ($key == 'user_id' && !empty($value)) {
                $user = get_user_by('ID', $value);

                if (!$user) {
                    $this->errors['data'][] = [
                        'name' => 'user',
                        'message' => 'کاربر وارد شده یافت نشد.'
                    ];

                    return false;
                }
            } elseif ($key == 'ip' && !empty($value)) {

                if (!filter_var($value, FILTER_VALIDATE_IP)) {
                    $this->errors['data'][] = [
                        'name' => 'ip',
                        'message' => 'آی پی وارد شده صحیح نمی باشد.'
                    ];

                    return false;
                }
            } elseif ($key == 'currency' && !empty($value)) {
                if (!in_array($value, $this->currencies)) {
                    $this->errors['data'][] = [
                        'name' => 'currency',
                        'message' => 'واحد مالی قابل قبول نیست.'
                    ];

                    return false;
                }
            } elseif ($key == 'gateway' && !empty($value)) {

                $gateway = $this->get_gateway($value);

                if (!$gateway) {
                    return false;
                }

            }

        }

        return true;
    }

    public function get_gateway($gateway_slug = null)
    {
        if (!is_null($gateway_slug)) {
            $this->gateway_default_slug = $gateway_slug;
        }

        if ($this->gateway && $this->gateway_default_slug == $this->gateway['slug']) {
            return $this->gateway;
        }

        $gateway = get_transient('mdr_gateway_' . $this->gateway_default_slug);

        if (empty($gateway)) {
            global $wpdb;

            $gateway_tb = new DBAccess($wpdb->prefix . 'mdr_gateways');
            $gateway = $gateway_tb->get_row(['slug' => $this->gateway_default_slug]);

        }

        if (empty($gateway)) {
            $this->errors['data'][] = [
                'name' => 'gateway',
                'message' => 'درگاه فعالی یافت نشد.'
            ];

            return false;
        }

        $this->gateway = $gateway;
        set_transient('mdr_gateway_' . $this->gateway_default_slug, $gateway);

        return $gateway;

    }

    public function complete_create_records_after_check_data()
    {

        if (!isset($this->transaction_data['user_id'])) {
            $user = mdr_get_or_create_user($this->inputs['phone'], $this->inputs['full_name']);
            $this->transaction_data['user_id'] = $user->ID;
        }

        if (!isset($this->transaction_data['inputs'])) {
            $this->transaction_data['inputs'] = $this->inputs;
        }

        if (!isset($this->transaction_data['ip'])) {
            $this->transaction_data['ip'] = mdr_get_user_ip();
        }

        if (!isset($this->transaction_data['amount'])) {
            $this->transaction_data['amount'] = 0;
        }

        if (!isset($this->transaction_data['gateway']) || !empty($this->transaction_data['gateway'])) {

            $gateway = $this->get_gateway();

            if (empty($gateway)) {

                return false;
            }

            $this->transaction_data['gateway'] = $this->gateway['slug'];

        }

        return true;

    }

    public array $transaction = [];
    public array $transaction_data = [];
    public array $inputs = [];
    public array $product = [];
    public array $errors = [];
    public array $gateways = [];
    public array $gateway = [];
    public array $currencies = ['IRR', 'IRT'];
    public string $currency = 'IRT';
    private string $gateway_default_slug = 'nabik';


}